//########################################################################
// (C)  Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

/*  ------------------------------------------------------------------------
    RBTree algorithm adopted from 
    http://eternallyconfuzzled.com/tuts/datastructures/jsw_tut_rbtree.aspx
    ------------------------------------------------------------------------

    Red Black balanced tree library

    > Created (Julienne Walker): August 23, 2003
    > Modified (Julienne Walker): March 14, 2008

    This code is in the public domain. Anyone may
    use it or change it in any way that they see
    fit. The author assumes no responsibility for 
    damages incurred through use of the original
    code or any variations thereof.

    It is requested, but not required, that due
    credit is given to the original author and
    anyone who has modified the code through
    a header comment, such as this one.
    ------------------------------------------------------------------------ */

#include <FeatStd/Diagnostics/Log.h>
#include "RBTree.h"

namespace FeatStd { namespace Internal { namespace RBTree {

    //FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdSystem);

/// @addtogroup FEATSTD_UTILS
/// @{

    // ========================================================================

    // ------------------------------------------------------------------------
    Node::Node()
    {
        FEATSTD_COMPILETIME_ASSERT((Int(NodeColor::Red) == 1) && (Int(NodeColor::Black) == 0));
        FEATSTD_COMPILETIME_ASSERT(sizeof(PtrValType) == sizeof(Node*));

        FEATSTD_LINT_NEXT_EXPRESSION(1938, "cRedFlag is a private constant declared in class Node")
        mLinkVal[0] = cRedFlag;
        mLinkVal[1] = 0;

#if defined(FEATSTD_DEBUG)
        // check here if 0 pointer has all bits zero - should be true for all
        // architectures
        union Value {
            Node *mLink;
            PtrValType mLinkVal;
        };

        Value value;
        value.mLink = 0;
        FEATSTD_DEBUG_ASSERT(value.mLinkVal == 0);
#endif
    }

    // ------------------------------------------------------------------------
    Node::Node(Node *left, Node *right)
    {
        FEATSTD_LINT_NEXT_EXPRESSION(1938, "cRedFlag is a private constant declared in class Node")
        mLinkVal[0] = PointerToScalar<PtrValType>(left) | cRedFlag;
        mLinkVal[1] = PointerToScalar<PtrValType>(right);
    }

    // ------------------------------------------------------------------------
    Node::~Node()
    {
        mLinkVal[0] = 0;
        mLinkVal[1] = 0;
    }

    // ========================================================================

    // ------------------------------------------------------------------------
    RedBlackTreeBase::RedBlackTreeBase() : mRoot(0)
    {
    }

    // ========================================================================

    // ------------------------------------------------------------------------
    void Functions::PaintItRed(Node *node)
    {
        FEATSTD_COMPILETIME_ASSERT((Int(Right) == 1) && (Int(Left) == 0));
        FEATSTD_DEBUG_ASSERT(node->GetLeftLink() != 0 && node->GetRightLink() != 0);

        node->SetColor(NodeColor::Red);
        node->GetLeftLink()->SetColor(NodeColor::Black);
        node->GetRightLink()->SetColor(NodeColor::Black);
    }

    // ------------------------------------------------------------------------
    void Functions::PaintItBlack(Node *node)
    {
        FEATSTD_DEBUG_ASSERT(node->GetLeftLink() != 0 && node->GetRightLink() != 0);

        node->SetColor(NodeColor::Black);
        node->GetLeftLink()->SetColor(NodeColor::Red);
        node->GetRightLink()->SetColor(NodeColor::Red);
    }

    // ------------------------------------------------------------------------
    Node* Functions::RotateSingle(Node *root, Direction dir)
    {
        FEATSTD_DEBUG_ASSERT(root != 0);

        Direction opposite = Opposite(dir);

        Node *save = root->GetLink(UInt32(opposite));

        root->SetLink(UInt32(opposite), save->GetLink(UInt32(dir)));
        save->SetLink(UInt32(dir), root);

        root->SetColor(NodeColor::Red);
        save->SetColor(NodeColor::Black);

        return save;
    }

    // ------------------------------------------------------------------------
    Node* Functions::RotateDouble(Node *root, Direction dir)
    {
        FEATSTD_DEBUG_ASSERT(root != 0);
        Direction opposite = Opposite(dir);
        root->SetLink(UInt32(opposite), RotateSingle(root->GetLink(UInt32(opposite)), opposite));
        return RotateSingle(root, dir );
    }

    // ------------------------------------------------------------------------
    void Functions::Insert(Node *&root, Node *node, CompareNodeSignature CompareFn)
    {
        if (root == 0) {
            // Empty tree case
            root = node;
        }
        else {
            Node head(0, root);       // False tree root

            Direction dir = Left;
            Direction last = Left;

            // Set up helpers
            Node *t = &head;
            Node *g = 0;
            Node *p = 0;
            Node *q = root;

            FEATSTD_LINT_SYMBOL(613, q, "q is checked for 0")
            FEATSTD_LINT_SYMBOL(613, p, "q is assigned to p, thus cannot be 0")

            // Search down the tree
            for (;;) {
                if (q == 0) {
                    // Insert new node at the bottom
                    p->SetLink(UInt32(dir), node);
                    q = node;
                }
                else if (IsRed(q->GetLeftLink()) && IsRed(q->GetRightLink())) {
                    // Color flip
                    PaintItRed(q);
                }
                else {
                    // intentionally left empty
                }

                // Fix red violation
                if (IsRed(q) && IsRed(p)) {
                    Direction dir2 = (t->GetRightLink() == g) ? Right : Left;

                    if (q == p->GetLink(UInt32(last))) {
                        t->SetLink(UInt32(dir2), RotateSingle(g, Opposite(last)));
                    }
                    else {
                        t->SetLink(UInt32(dir2), RotateDouble (g, Opposite(last)));
                    }
                }

                // if node has been inserted, q is pointing to node -> break
                if (q == node) {
                    break;
                }

                // Stop if found
                last = dir;
                dir = (CompareFn(*node, *q) <= 0) ? Left : Right;

                // Update helpers
                if (g != 0) {
                    t = g;
                }
                g = p;
                p = q;
                q = q->GetLink(UInt32(dir));
            }
            // Update root
            root = head.GetRightLink();
        }
        // Make root black
        root->SetColor(NodeColor::Black);
    }


    // ------------------------------------------------------------------------
    Node* Functions::Remove(Node *&root, NodeSelector &selector)
    {
        if (root == 0) {
            return 0;
        }

        Node head(0, root);            // False tree root
        Node *selectedNode = 0;         // Found item
        Node *selectedNodeParent = 0;
        Direction dir = Right;
        Direction oppositeDir = Left;
        
        // Set up helpers
        Node *q = &head;
        Node *g = 0;
        Node *p = 0;

        FEATSTD_LINT_SYMBOL(613, p, "p is checked for 0")
        FEATSTD_LINT_SYMBOL(613, g, "g receives p which is checked")

        // Search and push a red down
        while (q->GetLink(UInt32(dir)) != 0) {
            Direction last = dir;
            Direction oppositeLast = oppositeDir;

            // Update helpers
            g = p;
            p = q;
            q = q->GetLink(UInt32(dir));
            Int32 rc = selector(*q);
            if (rc <= 0) {
                dir = Left;
                oppositeDir = Right;
            }
            else {
                dir = Right;
                oppositeDir = Left;
            }

            // Save found node
            if (rc == 0) {
                selectedNode = q;
                selectedNodeParent = p;
            }

            // Push the red node down
            if ((!IsRed(q)) && (!IsRed(q->GetLink(UInt32(dir))))) {
                if (IsRed(q->GetLink(UInt32(oppositeDir)))) {
                    p->SetLink(UInt32(last), RotateSingle(q, dir));
                    p = p->GetLink(UInt32(last));
                    if (q == selectedNode) {
                        // if selectedNode got rotated, update selectedNodeParent too
                        selectedNodeParent = p;
                    }
                }
                else if (!IsRed(q->GetLink(UInt32(oppositeDir)))) {
                    Node *s = p->GetLink(UInt32(oppositeLast));

                    if (s != 0) {
                        if ((!IsRed(s->GetLink(UInt32(oppositeLast)))) && (!IsRed(s->GetLink(UInt32(last))))) {
                            // Color flip
                            PaintItBlack(p);
                        }
                        else {
                            Direction dir2 = (g->GetRightLink() == p) ? Right : Left;

                            if (IsRed(s->GetLink(UInt32(last)))) {
                                g->SetLink(UInt32(dir2), RotateDouble(p, last));
                                if (p == selectedNode) {
                                    // if selectedNode got rotated, update selectedNodeParent too
                                    selectedNodeParent = g->GetLink(UInt32(dir2));
                                }
                            }
                            else if(IsRed(s->GetLink(UInt32(oppositeLast)))) {
                                g->SetLink(UInt32(dir2), RotateSingle(p, last));
                                if (p == selectedNode) {
                                    // if selectedNode got rotated, update selectedNodeParent too
                                    selectedNodeParent = g->GetLink(UInt32(dir2));
                                }
                            }
                            else {
                                // intentionally left empty
                            }

                            // Ensure correct coloring
                            q->SetColor(NodeColor::Red);
                            PaintItRed(g->GetLink(UInt32(dir2)));
                        }
                    }
                }
                else {
                    // intentionally left empty
                }
            }
        }

        // Replace and remove if found 
        if (selectedNode != 0) {
            //selectedNode->data = q->data;
            // unlink q
            FEATSTD_LINT_NEXT_EXPRESSION(794, "p is checked and can not be 0")
            p->SetLink((p->GetLeftLink() == q) ? 0 : 1, q->GetLink((q->GetLeftLink() == 0) ? 1 : 0));
            if (q != selectedNode) {
                FEATSTD_DEBUG_ASSERT(selectedNodeParent != 0);
                FEATSTD_DEBUG_ASSERT(selectedNodeParent->GetLeftLink() == selectedNode || selectedNodeParent->GetRightLink() == selectedNode);

                q->SetLink(0, selectedNode->GetLeftLink());
                q->SetLink(1, selectedNode->GetRightLink());
                q->SetColor(selectedNode->GetColor());

                FEATSTD_LINT_NEXT_EXPRESSION(613, "selectedNodeParent can not be null if selectedNode != 0")
                selectedNodeParent->SetLink((selectedNodeParent->GetLeftLink() == selectedNode) ? 0 : 1, q);
            }
        }

        // Update root and make it black
        root = head.GetRightLink();
        if (root != 0) {
            root->SetColor(NodeColor::Black);
        }

        return selectedNode;
    }

    // ------------------------------------------------------------------------
    Node* Functions::Find(Node *root, NodeSelector &selector)
    {
        Node *it = root;
        for (;;) {
            if (it == 0) {
                break;
            }

            Int32 rc = selector(*it);
            if (rc == 0 ) {
                break;
            }
            it = it->GetLink(UInt32((rc <= 0) ? Left : Right));
        }
        return it;
    }

    // ------------------------------------------------------------------------
    static bool StepDownLeft(Node *&node, Node *&parent)
    {
        FEATSTD_DEBUG_ASSERT(node != 0);

        Node *tmp = node->GetLeftLink();
        if (tmp != 0) {
            FEATSTD_DEBUG_ASSERT(!node->IsTagged());

            node->SetLeftLink(parent);
            parent = node;
            node = tmp;
        }
        return tmp != 0;
    }

    // ------------------------------------------------------------------------
    static bool StepDownRight(Node *&node, Node *&parent)
    {
        FEATSTD_DEBUG_ASSERT(node != 0);
        Node *tmp = node->GetRightLink();
        if (tmp != 0) {
            node->SetRightLink(parent);
            node->SetTag();
            parent = node;
            node = tmp;
        }
        return tmp != 0;
    }

    // ------------------------------------------------------------------------
    static bool StepUp(Node *&node, Node *&parent)
    {
        FEATSTD_DEBUG_ASSERT(node != 0);

        if (parent == 0) {
            return false;
        }
        Node *grandParent;
        if (parent->IsTagged()) {
            grandParent = parent->GetRightLink();
            parent->SetRightLink(node);
            parent->SetTag(false);
        }
        else {
            grandParent = parent->GetLeftLink();
            parent->SetLeftLink(node);
        }
        node = parent;
        parent = grandParent;
        return true;
    }

    // ------------------------------------------------------------------------
    void Functions::UnlinkAllNodes(Node *&root, VisitSignature VisitFn, void *userData)
    {
        FEATSTD_DEBUG_ASSERT(VisitFn != 0);
        if ((VisitFn == 0) || (root == 0)) {
            return;
        }

        Node *node = root;
        Node *parent = 0;

        do {
            while (StepDownLeft(node, parent)) {
            }

            for (;;) {
                if (!node->IsLeave()) {
                    node->SetLeftLink(parent);
                    parent = node;
                    node = node->GetRightLink();
                    break;
                }
                (void) VisitFn(*node, userData);
                if (parent == 0) {
                    break;
                }
                if ((parent->GetRightLink() == node) || (parent->GetRightLink() == 0)) {
                    node = parent;
                    parent = node->GetLeftLink();
                    node->Reset();
                }
                else {
                    node = parent->GetRightLink();
                    break;
                }
            }
        } while (parent != 0);
            
        root = 0;
    }

    // ------------------------------------------------------------------------
    bool Functions::Visit(Node *node, VisitSignature VisitFn, void *userData)
    {
        FEATSTD_DEBUG_ASSERT(VisitFn != 0);
        if ((VisitFn == 0) || (node == 0)) {
            return true;
        }

        Node *parent = 0;
        bool abandoned = false;
        bool done = false;

        do {
            while (StepDownLeft(node, parent)) {
            }

            bool tagged = false;
            for (;;) {
                if (!tagged) {
                    if (!VisitFn(*node, userData)) {
                        abandoned = true;
                        break;
                    }
                    if (StepDownRight(node, parent)) {
                        break;
                    }
                }
                tagged = (parent != 0) && (parent->IsTagged());
                if (!StepUp(node, parent)) {
                    done = true;
                    break;
                }
            }
        } while ((!done) && (!abandoned));
            
        if (abandoned) {
            while (StepUp(node, parent)) {
            }
        }

        return !abandoned;
    }

/// @}
}}}
